<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../src/TokenMonitor.php';
require_once __DIR__ . '/../src/PriceAnalyzer.php';
require_once __DIR__ . '/../src/RiskAnalyzer.php';
require_once __DIR__ . '/../src/SocialAnalyzer.php';
require_once __DIR__ . '/../src/TelegramBot.php';

use TokenMonitor\TokenMonitor;
use TokenMonitor\PriceAnalyzer;
use TokenMonitor\RiskAnalyzer;
use TokenMonitor\SocialAnalyzer;
use TokenMonitor\TelegramBot;

class DailyAnalyzer {
    private $db;
    private $config;
    private $token_monitor;
    private $price_analyzer;
    private $risk_analyzer;
    private $social_analyzer;
    private $telegram_bot;
    private $log_file;

    public function __construct() {
        $this->log_file = __DIR__ . '/../logs/analyze_' . date('Y-m-d') . '.log';
        $this->initializeLogger();
        
        try {
            $this->loadConfig();
            $this->connectDatabase();
            $this->initializeComponents();
        } catch (\Exception $e) {
            $this->log("CRITICAL ERROR: " . $e->getMessage());
            die("Initialization failed: " . $e->getMessage());
        }
    }

    // اجرای تحلیل روزانه
    public function run() {
        $this->log("Starting daily analysis...");

        try {
            // دریافت آمار کلی بازار
            $market_stats = $this->getMarketStats();

            // تحلیل روند بازار
            $market_trend = $this->analyzeMarketTrend();

            // بررسی توکن‌های برتر
            $top_performers = $this->analyzeTopPerformers();

            // تحلیل ریسک‌های سیستمی
            $systemic_risks = $this->analyzeSystemicRisks();

            // جمع‌بندی و ارسال گزارش
            $this->sendDailySummary([
                'market_stats' => $market_stats,
                'market_trend' => $market_trend,
                'top_performers' => $top_performers,
                'systemic_risks' => $systemic_risks
            ]);

            $this->log("Daily analysis completed successfully.");
        } catch (\Exception $e) {
            $this->log("ERROR during analysis: " . $e->getMessage());
            // ارسال نوتیفیکیشن خطا به ادمین
            $this->telegram_bot->sendMessage(
                "❌ Error in daily analysis: " . $e->getMessage(),
                $this->config['admin_chat_id']
            );
        }
    }

    // دریافت آمار کلی بازار
    private function getMarketStats() {
        $this->log("Collecting market statistics...");
        
        $stats = [
            'total_volume' => 0,
            'total_liquidity' => 0,
            'new_tokens' => 0,
            'active_tokens' => 0,
            'volume_change' => 0,
            'liquidity_change' => 0
        ];

        // دریافت داده‌های 24 ساعت گذشته
        $stmt = $this->db->prepare("
            SELECT 
                SUM(volume_24h) as total_volume,
                SUM(liquidity) as total_liquidity,
                COUNT(*) as active_tokens,
                SUM(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 ELSE 0 END) as new_tokens
            FROM tokens 
            WHERE status = 'active'
        ");
        $stmt->execute();
        $current_data = $stmt->fetch(PDO::FETCH_ASSOC);

        // دریافت داده‌های روز قبل برای مقایسه
        $stmt = $this->db->prepare("
            SELECT 
                SUM(volume_24h) as total_volume,
                SUM(liquidity) as total_liquidity
            FROM token_history
            WHERE DATE(timestamp) = DATE_SUB(CURDATE(), INTERVAL 1 DAY)
        ");
        $stmt->execute();
        $previous_data = $stmt->fetch(PDO::FETCH_ASSOC);

        // محاسبه تغییرات
        if ($previous_data['total_volume'] > 0) {
            $stats['volume_change'] = (($current_data['total_volume'] - $previous_data['total_volume']) 
                                    / $previous_data['total_volume']) * 100;
        }
        if ($previous_data['total_liquidity'] > 0) {
            $stats['liquidity_change'] = (($current_data['total_liquidity'] - $previous_data['total_liquidity']) 
                                       / $previous_data['total_liquidity']) * 100;
        }

        $stats = array_merge($stats, [
            'total_volume' => $current_data['total_volume'],
            'total_liquidity' => $current_data['total_liquidity'],
            'active_tokens' => $current_data['active_tokens'],
            'new_tokens' => $current_data['new_tokens']
        ]);

        return $stats;
    }

    // تحلیل روند بازار
    private function analyzeMarketTrend() {
        $this->log("Analyzing market trend...");
        
        // دریافت داده‌های هفته گذشته
        $stmt = $this->db->prepare("
            SELECT 
                DATE(timestamp) as date,
                SUM(volume_24h) as daily_volume,
                SUM(liquidity) as daily_liquidity,
                COUNT(DISTINCT token_address) as active_tokens
            FROM token_history
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            GROUP BY DATE(timestamp)
            ORDER BY date ASC
        ");
        $stmt->execute();
        $weekly_data = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // تحلیل روند
        $trend = [
            'volume_trend' => $this->calculateTrend(array_column($weekly_data, 'daily_volume')),
            'liquidity_trend' => $this->calculateTrend(array_column($weekly_data, 'daily_liquidity')),
            'tokens_trend' => $this->calculateTrend(array_column($weekly_data, 'active_tokens')),
            'market_sentiment' => 'neutral'
        ];

        // محاسبه sentiment بازار
        $positive_factors = 0;
        if ($trend['volume_trend'] === 'uptrend') $positive_factors++;
        if ($trend['liquidity_trend'] === 'uptrend') $positive_factors++;
        if ($trend['tokens_trend'] === 'uptrend') $positive_factors++;

        if ($positive_factors >= 2) {
            $trend['market_sentiment'] = 'bullish';
        } elseif ($positive_factors <= 1) {
            $trend['market_sentiment'] = 'bearish';
        }

        return $trend;
    }

    // تحلیل توکن‌های برتر
    private function analyzeTopPerformers() {
        $this->log("Analyzing top performing tokens...");
        
        $categories = [
            'price_gainers' => $this->getTopTokensByMetric('price_change_24h', 'DESC', 5),
            'volume_gainers' => $this->getTopTokensByMetric('volume_24h', 'DESC', 5),
            'new_trending' => $this->getTopNewTokens(5)
        ];

        foreach ($categories as $category => $tokens) {
            foreach ($tokens as &$token) {
                // افزودن تحلیل تکنیکال
                $token['technical_analysis'] = $this->price_analyzer->analyzePriceAction($token['token_address']);
                
                // افزودن ارزیابی ریسک
                $token['risk_assessment'] = $this->risk_analyzer->analyzeRisks($token);
                
                // افزودن تحلیل اجتماعی
                $token['social_analysis'] = $this->social_analyzer->analyzeSocialPresence($token);
            }
        }

        return $categories;
    }

    // تحلیل ریسک‌های سیستمی
    private function analyzeSystemicRisks() {
        $this->log("Analyzing systemic risks...");
        
        $risks = [];

        // بررسی تمرکز لیکوئیدیتی
        $liquidity_concentration = $this->analyzeLiquidityConcentration();
        if ($liquidity_concentration > 50) {
            $risks[] = [
                'type' => 'liquidity_concentration',
                'severity' => 'high',
                'description' => "High liquidity concentration in top pairs"
            ];
        }

        // بررسی نوسانات غیرعادی
        $unusual_volatility = $this->checkUnusualVolatility();
        if ($unusual_volatility) {
            $risks[] = [
                'type' => 'market_volatility',
                'severity' => 'medium',
                'description' => "Unusual market volatility detected"
            ];
        }

        // بررسی الگوهای مشکوک معاملات
        $suspicious_patterns = $this->detectSuspiciousTrading();
        if (!empty($suspicious_patterns)) {
            $risks[] = [
                'type' => 'suspicious_trading',
                'severity' => 'high',
                'description' => "Suspicious trading patterns detected",
                'details' => $suspicious_patterns
            ];
        }

        return $risks;
    }

    // ارسال گزارش روزانه
    private function sendDailySummary($data) {
        $message = $this->formatDailySummary($data);
        $this->telegram_bot->sendMessage($message);
        
        // ذخیره گزارش در دیتابیس
        $this->saveDailyReport($data);
    }

    // فرمت‌بندی گزارش روزانه
    private function formatDailySummary($data) {
        $message = "📊 <b>تحلیل روزانه بازار سولانا</b>\n\n";

        // آمار کلی
        $message .= "📈 <b>آمار کلی:</b>\n";
        $message .= "• حجم کل معاملات: $" . number_format($data['market_stats']['total_volume']) . "\n";
        $message .= "• تغییر حجم: " . number_format($data['market_stats']['volume_change'], 2) . "%\n";
        $message .= "• لیکوئیدیتی کل: $" . number_format($data['market_stats']['total_liquidity']) . "\n";
        $message .= "• توکن‌های جدید: {$data['market_stats']['new_tokens']}\n\n";

        // روند بازار
        $message .= "🎯 <b>روند بازار:</b>\n";
        $message .= "• سنتیمنت: " . $this->getSentimentEmoji($data['market_trend']['market_sentiment']) . " ";
        $message .= ucfirst($data['market_trend']['market_sentiment']) . "\n";
        $message .= "• روند حجم: " . $this->getTrendEmoji($data['market_trend']['volume_trend']) . "\n";
        $message .= "• روند لیکوئیدیتی: " . $this->getTrendEmoji($data['market_trend']['liquidity_trend']) . "\n\n";

        // برترین‌ها
        $message .= "🏆 <b>برترین عملکردها:</b>\n";
        foreach ($data['top_performers']['price_gainers'] as $token) {
            $message .= "• {$token['symbol']}: +" . number_format($token['price_change_24h'], 2) . "%\n";
        }

        // ریسک‌ها
        if (!empty($data['systemic_risks'])) {
            $message .= "\n⚠️ <b>هشدارهای ریسک:</b>\n";
            foreach ($data['systemic_risks'] as $risk) {
                $message .= "• {$risk['description']}\n";
            }
        }

        return $message;
    }

    // محاسبه روند
    private function calculateTrend($data) {
        $changes = [];
        for ($i = 1; $i < count($data); $i++) {
            $changes[] = $data[$i] - $data[$i-1];
        }
        
        $positive_changes = count(array_filter($changes, function($change) {
            return $change > 0;
        }));
        
        $trend_ratio = $positive_changes / count($changes);
        
        if ($trend_ratio > 0.6) {
            return 'uptrend';
        } elseif ($trend_ratio < 0.4) {
            return 'downtrend';
        }
        return 'sideways';
    }

    // دریافت ایموجی روند
    private function getTrendEmoji($trend) {
        switch ($trend) {
            case 'uptrend':
                return '📈 Uptrend';
            case 'downtrend':
                return '📉 Downtrend';
            default:
                return '➡️ Sideways';
        }
    }

    // دریافت ایموجی سنتیمنت
    private function getSentimentEmoji($sentiment) {
        switch ($sentiment) {
            case 'bullish':
                return '🟢';
            case 'bearish':
                return '🔴';
            default:
                return '🟡';
        }
    }

    // دریافت توکن‌های برتر بر اساس معیار
    private function getTopTokensByMetric($metric, $order = 'DESC', $limit = 5) {
        $stmt = $this->db->prepare("
            SELECT * FROM tokens 
            WHERE status = 'active' 
            ORDER BY {$metric} {$order}
            LIMIT {$limit}
        ");
        $stmt->execute();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // دریافت توکن‌های جدید برتر
    private function getTopNewTokens($limit = 5) {
        $stmt = $this->db->prepare("
            SELECT * FROM tokens 
            WHERE status = 'active' 
            AND created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR)
            ORDER BY volume_24h DESC
            LIMIT ?
        ");
        $stmt->execute([$limit]);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // تحلیل تمرکز لیکوئیدیتی
    private function analyzeLiquidityConcentration() {
        $stmt = $this->db->prepare("
            SELECT 
                SUM(liquidity) as total_liquidity,
                (
                    SELECT SUM(liquidity) 
                    FROM tokens 
                    WHERE status = 'active'
                    ORDER BY liquidity DESC 
                    LIMIT 5
                ) as top_5_liquidity
            FROM tokens 
            WHERE status = 'active'
        ");
        $stmt->execute();
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($result['total_liquidity'] > 0) {
            return ($result['top_5_liquidity'] / $result['total_liquidity']) * 100;
        }
        return 0;
    }

    // بررسی نوسانات غیرعادی
    private function checkUnusualVolatility() {
        $stmt = $this->db->prepare("
            SELECT 
                AVG(ABS(price_change_24h)) as avg_volatility,
                STDDEV(price_change_24h) as volatility_stddev
            FROM tokens 
            WHERE status = 'active'
            AND created_at < DATE_SUB(NOW(), INTERVAL 24 HOUR)
        ");
        $stmt->execute();
        $stats = $stmt->fetch(PDO::FETCH_ASSOC);
        
        // اگر میانگین تغییرات بیش از 30% باشد یا انحراف معیار بیش از 50%
        return $stats['avg_volatility'] > 30 || $stats['volatility_stddev'] > 50;
    }

    // شناسایی الگوهای مشکوک معاملات
    private function detectSuspiciousTrading() {
        $patterns = [];
        
        // بررسی تغییرات قیمت ناگهانی
        $stmt = $this->db->prepare("
            SELECT token_address, symbol, price_change_24h
            FROM tokens
            WHERE status = 'active' 
            AND ABS(price_change_24h) > 100
        ");
        $stmt->execute();
        $sudden_changes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (!empty($sudden_changes)) {
            $patterns[] = [
                'type' => 'sudden_price_change',
                'tokens' => $sudden_changes
            ];
        }

        // بررسی حجم معاملات غیرعادی
        $stmt = $this->db->prepare("
            SELECT token_address, symbol, volume_24h
            FROM tokens
            WHERE status = 'active'
            AND volume_24h > (
                SELECT AVG(volume_24h) * 5
                FROM tokens
                WHERE status = 'active'
            )
        ");
        $stmt->execute();
        $unusual_volumes = $stmt->fetchAll(PDO::FETCH_ASSOC);
        
        if (!empty($unusual_volumes)) {
            $patterns[] = [
                'type' => 'unusual_volume',
                'tokens' => $unusual_volumes
            ];
        }

        return $patterns;
    }

    // ذخیره گزارش روزانه
    private function saveDailyReport($data) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO daily_reports (
                    date,
                    market_stats,
                    market_trend,
                    top_performers,
                    systemic_risks,
                    created_at
                ) VALUES (
                    CURDATE(),
                    ?,
                    ?,
                    ?,
                    ?,
                    NOW()
                )
            ");
            
            $stmt->execute([
                json_encode($data['market_stats']),
                json_encode($data['market_trend']),
                json_encode($data['top_performers']),
                json_encode($data['systemic_risks'])
            ]);
            
            $this->log("Daily report saved successfully.");
            return true;
        } catch (\PDOException $e) {
            $this->log("Error saving daily report: " . $e->getMessage());
            return false;
        }
    }

    // راه‌اندازی لاگر
    private function initializeLogger() {
        if (!file_exists(dirname($this->log_file))) {
            mkdir(dirname($this->log_file), 0777, true);
        }
    }

    // لود کردن کانفیگ
    private function loadConfig() {
        $config_file = __DIR__ . '/../config/config.php';
        if (!file_exists($config_file)) {
            throw new \Exception("Config file not found!");
        }
        $this->config = require $config_file;
    }

    // اتصال به دیتابیس
    private function connectDatabase() {
        try {
            $this->db = new PDO(
                "mysql:host={$this->config['database']['host']};dbname={$this->config['database']['dbname']};charset=utf8mb4",
                $this->config['database']['user'],
                $this->config['database']['pass'],
                [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
            );
        } catch (\PDOException $e) {
            throw new \Exception("Database connection failed: " . $e->getMessage());
        }
    }

    // راه‌اندازی کامپوننت‌ها
    private function initializeComponents() {
        $this->token_monitor = new TokenMonitor($this->db, $this->config);
        $this->price_analyzer = new PriceAnalyzer($this->db, $this->config);
        $this->risk_analyzer = new RiskAnalyzer($this->db, $this->config);
        $this->social_analyzer = new SocialAnalyzer($this->db, $this->config);
        $this->telegram_bot = new TelegramBot($this->config, $this->db);
    }

    // لاگ کردن پیام‌ها
    private function log($message) {
        $log_message = "[" . date('Y-m-d H:i:s') . "] " . $message . "\n";
        file_put_contents($this->log_file, $log_message, FILE_APPEND);
    }
}

// اجرای تحلیل
try {
    $analyzer = new DailyAnalyzer();
    $analyzer->run();
} catch (\Exception $e) {
    error_log("Critical error in analyze.php: " . $e->getMessage());
}