<?php
namespace TokenMonitor;

class PriceAnalyzer {
    private $db;
    private $config;

    public function __construct($db, $config) {
        $this->db = $db;
        $this->config = $config;
    }

    // تحلیل کامل قیمت
    public function analyzePriceAction($token_address) {
        $price_data = $this->getPriceHistory($token_address);
        
        $analysis = [
            'technical' => $this->technicalAnalysis($price_data),
            'volatility' => $this->calculateVolatility($price_data),
            'momentum' => $this->calculateMomentum($price_data),
            'support_resistance' => $this->findSupportResistance($price_data),
            'patterns' => $this->detectPatterns($price_data)
        ];
        
        // بررسی سیگنال‌های خرید/فروش
        $signals = $this->generateSignals($analysis);
        
        return [
            'analysis' => $analysis,
            'signals' => $signals
        ];
    }

    // دریافت تاریخچه قیمت
    private function getPriceHistory($token_address, $timeframe = '1d', $limit = 100) {
        try {
            $stmt = $this->db->prepare("
                SELECT * FROM price_history 
                WHERE token_address = ? AND timeframe = ?
                ORDER BY timestamp DESC 
                LIMIT ?
            ");
            $stmt->execute([$token_address, $timeframe, $limit]);
            return $stmt->fetchAll(\PDO::FETCH_ASSOC);
        } catch (\PDOException $e) {
            error_log("Error fetching price history: " . $e->getMessage());
            return [];
        }
    }

    // تحلیل تکنیکال
    private function technicalAnalysis($price_data) {
        return [
            'moving_averages' => $this->calculateMovingAverages($price_data),
            'rsi' => $this->calculateRSI($price_data),
            'macd' => $this->calculateMACD($price_data),
            'bollinger_bands' => $this->calculateBollingerBands($price_data),
            'volume_analysis' => $this->analyzeVolume($price_data)
        ];
    }

    // محاسبه میانگین‌های متحرک
    private function calculateMovingAverages($price_data) {
        $closes = array_column($price_data, 'close');
        
        return [
            'sma_20' => $this->calculateSMA($closes, 20),
            'sma_50' => $this->calculateSMA($closes, 50),
            'sma_200' => $this->calculateSMA($closes, 200),
            'ema_20' => $this->calculateEMA($closes, 20),
            'ema_50' => $this->calculateEMA($closes, 50)
        ];
    }

    // محاسبه SMA
    private function calculateSMA($data, $period) {
        $sma = [];
        for ($i = 0; $i < count($data); $i++) {
            if ($i < $period - 1) {
                $sma[] = null;
                continue;
            }
            
            $sum = 0;
            for ($j = 0; $j < $period; $j++) {
                $sum += $data[$i - $j];
            }
            $sma[] = $sum / $period;
        }
        return $sma;
    }

    // محاسبه EMA
    private function calculateEMA($data, $period) {
        $multiplier = 2 / ($period + 1);
        $ema = [];
        
        // اولین EMA برابر با SMA است
        $sma = array_sum(array_slice($data, 0, $period)) / $period;
        $ema[] = $sma;
        
        for ($i = 1; $i < count($data); $i++) {
            $ema[] = ($data[$i] - $ema[$i-1]) * $multiplier + $ema[$i-1];
        }
        
        return $ema;
    }

    // محاسبه RSI
    private function calculateRSI($price_data, $period = 14) {
        $closes = array_column($price_data, 'close');
        $gains = [];
        $losses = [];
        
        // محاسبه gains و losses
        for ($i = 1; $i < count($closes); $i++) {
            $change = $closes[$i] - $closes[$i-1];
            if ($change >= 0) {
                $gains[] = $change;
                $losses[] = 0;
            } else {
                $gains[] = 0;
                $losses[] = abs($change);
            }
        }
        
        // محاسبه average gains و losses
        $avg_gain = array_sum(array_slice($gains, 0, $period)) / $period;
        $avg_loss = array_sum(array_slice($losses, 0, $period)) / $period;
        
        // محاسبه RS و RSI
        $rs = $avg_loss == 0 ? 100 : $avg_gain / $avg_loss;
        $rsi = 100 - (100 / (1 + $rs));
        
        return $rsi;
    }

    // محاسبه MACD
    private function calculateMACD($price_data) {
        $closes = array_column($price_data, 'close');
        
        $ema12 = $this->calculateEMA($closes, 12);
        $ema26 = $this->calculateEMA($closes, 26);
        
        // خط MACD
        $macd_line = array_map(function($a, $b) {
            return $a - $b;
        }, $ema12, $ema26);
        
        // خط سیگنال (EMA 9 روی MACD)
        $signal_line = $this->calculateEMA($macd_line, 9);
        
        // هیستوگرام
        $histogram = array_map(function($a, $b) {
            return $a - $b;
        }, $macd_line, $signal_line);
        
        return [
            'macd_line' => $macd_line,
            'signal_line' => $signal_line,
            'histogram' => $histogram
        ];
    }

    // محاسبه باندهای بولینگر
    private function calculateBollingerBands($price_data, $period = 20) {
        $closes = array_column($price_data, 'close');
        
        // محاسبه SMA
        $sma = $this->calculateSMA($closes, $period);
        
        // محاسبه انحراف معیار
        $std_dev = [];
        for ($i = 0; $i < count($closes); $i++) {
            if ($i < $period - 1) {
                $std_dev[] = null;
                continue;
            }
            
            $slice = array_slice($closes, $i - $period + 1, $period);
            $variance = array_sum(array_map(function($x) use ($sma, $i) {
                return pow($x - $sma[$i], 2);
            }, $slice)) / $period;
            
            $std_dev[] = sqrt($variance);
        }
        
        // محاسبه باندها
        $upper_band = [];
        $lower_band = [];
        for ($i = 0; $i < count($sma); $i++) {
            if ($sma[$i] === null) {
                $upper_band[] = null;
                $lower_band[] = null;
                continue;
            }
            $upper_band[] = $sma[$i] + (2 * $std_dev[$i]);
            $lower_band[] = $sma[$i] - (2 * $std_dev[$i]);
        }
        
        return [
            'middle_band' => $sma,
            'upper_band' => $upper_band,
            'lower_band' => $lower_band
        ];
    }

    // تشخیص الگوهای قیمتی
    private function detectPatterns($price_data) {
        $patterns = [];
        
        if ($this->isDoubleBottom($price_data)) {
            $patterns[] = 'double_bottom';
        }
        
        if ($this->isDoubleTop($price_data)) {
            $patterns[] = 'double_top';
        }
        
        if ($this->isHeadAndShoulders($price_data)) {
            $patterns[] = 'head_and_shoulders';
        }
        
        if ($this->isBullishFlag($price_data)) {
            $patterns[] = 'bullish_flag';
        }
        
        return $patterns;
    }

    // تشخیص الگوی کف دوقلو
    private function isDoubleBottom($price_data) {
        // پیاده‌سازی الگوریتم تشخیص کف دوقلو
        return false; // فعلاً
    }

    // تولید سیگنال‌های معاملاتی
    private function generateSignals($analysis) {
        $signals = [];
        
        // بررسی سیگنال‌های تکنیکال
        if ($this->isBullishMACD($analysis['technical']['macd'])) {
            $signals[] = [
                'type' => 'buy',
                'source' => 'macd',
                'strength' => 'medium'
            ];
        }
        
        if ($this->isOversold($analysis['technical']['rsi'])) {
            $signals[] = [
                'type' => 'buy',
                'source' => 'rsi',
                'strength' => 'strong'
            ];
        }
        
        // بررسی الگوهای قیمتی
        foreach ($analysis['patterns'] as $pattern) {
            if (in_array($pattern, ['double_bottom', 'bullish_flag'])) {
                $signals[] = [
                    'type' => 'buy',
                    'source' => 'pattern_' . $pattern,
                    'strength' => 'strong'
                ];
            }
        }
        
        return $signals;
    }
}