<?php
require_once __DIR__ . '/../config/config.php';
require_once __DIR__ . '/../src/TokenMonitor.php';
require_once __DIR__ . '/../src/PriceAnalyzer.php';
require_once __DIR__ . '/../src/RiskAnalyzer.php';
require_once __DIR__ . '/../src/SocialAnalyzer.php';

session_start();

// تنظیمات اولیه
$config = require __DIR__ . '/../config/config.php';
try {
    $db = new PDO(
        "mysql:host={$config['database']['host']};dbname={$config['database']['dbname']};charset=utf8mb4",
        $config['database']['user'],
        $config['database']['pass'],
        [PDO::ATTR_ERRMODE => PDO::ERRMODE_EXCEPTION]
    );
} catch (PDOException $e) {
    die("Connection failed: " . $e->getMessage());
}

// دریافت آمار کلی
function getOverallStats($db) {
    $stmt = $db->query("
        SELECT 
            COUNT(*) as total_tokens,
            SUM(volume_24h) as total_volume,
            SUM(liquidity) as total_liquidity,
            COUNT(CASE WHEN created_at >= DATE_SUB(NOW(), INTERVAL 24 HOUR) THEN 1 END) as new_tokens_24h
        FROM tokens 
        WHERE status = 'active'
    ");
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

// دریافت سیگنال‌های اخیر
function getRecentSignals($db, $limit = 10) {
    $stmt = $db->prepare("
        SELECT s.*, t.symbol, t.name
        FROM signals s
        JOIN tokens t ON s.token_address = t.token_address
        ORDER BY s.created_at DESC
        LIMIT ?
    ");
    $stmt->execute([$limit]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// دریافت توکن‌های برتر
function getTopTokens($db, $metric = 'volume_24h', $limit = 5) {
    $stmt = $db->prepare("
        SELECT *
        FROM tokens
        WHERE status = 'active'
        ORDER BY {$metric} DESC
        LIMIT ?
    ");
    $stmt->execute([$limit]);
    return $stmt->fetchAll(PDO::FETCH_ASSOC);
}

// دریافت گزارش روزانه آخر
function getLatestDailyReport($db) {
    $stmt = $db->query("
        SELECT *
        FROM daily_reports
        ORDER BY date DESC
        LIMIT 1
    ");
    return $stmt->fetch(PDO::FETCH_ASSOC);
}

$stats = getOverallStats($db);
$recent_signals = getRecentSignals($db);
$top_volume = getTopTokens($db, 'volume_24h');
$top_gainers = getTopTokens($db, 'price_change_24h');
$latest_report = getLatestDailyReport($db);
?>

<!DOCTYPE html>
<html lang="fa" dir="rtl">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>داشبورد مانیتورینگ توکن‌های سولانا</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/@fortawesome/fontawesome-free@6.0.0/css/all.min.css">
</head>
<body class="bg-gray-100">
    <div class="min-h-screen">
        <!-- هدر -->
        <nav class="bg-green-600 text-white shadow-lg">
            <div class="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8">
                <div class="flex items-center justify-between h-16">
                    <div class="flex items-center">
                        <div class="flex-shrink-0">
                            <h1 class="text-xl font-bold">مانیتورینگ توکن‌های سولانا</h1>
                        </div>
                    </div>
                </div>
            </div>
        </nav>

        <!-- محتوای اصلی -->
        <main class="max-w-7xl mx-auto py-6 sm:px-6 lg:px-8">
            <!-- آمار کلی -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-semibold text-gray-700 mb-2">کل توکن‌ها</h3>
                    <p class="text-3xl font-bold text-green-600"><?php echo number_format($stats['total_tokens']); ?></p>
                </div>
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-semibold text-gray-700 mb-2">حجم 24 ساعته</h3>
                    <p class="text-3xl font-bold text-green-600">$<?php echo number_format($stats['total_volume']); ?></p>
                </div>
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-semibold text-gray-700 mb-2">لیکوئیدیتی کل</h3>
                    <p class="text-3xl font-bold text-green-600">$<?php echo number_format($stats['total_liquidity']); ?></p>
                </div>
                <div class="bg-white rounded-lg shadow p-6">
                    <h3 class="text-lg font-semibold text-gray-700 mb-2">توکن‌های جدید</h3>
                    <p class="text-3xl font-bold text-green-600"><?php echo number_format($stats['new_tokens_24h']); ?></p>
                </div>
            </div>

            <!-- سیگنال‌های اخیر و توکن‌های برتر -->
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                <!-- سیگنال‌های اخیر -->
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6">
                        <h2 class="text-xl font-bold text-gray-800 mb-4">سیگنال‌های اخیر</h2>
                        <div class="overflow-x-auto">
                            <table class="min-w-full">
                                <thead>
                                    <tr class="text-right bg-gray-50">
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">توکن</th>
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">نوع</th>
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">قیمت</th>
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">زمان</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-200">
                                    <?php foreach ($recent_signals as $signal): ?>
                                    <tr>
                                        <td class="px-6 py-4 text-sm text-gray-900"><?php echo htmlspecialchars($signal['symbol']); ?></td>
                                        <td class="px-6 py-4 text-sm">
                                            <span class="px-2 inline-flex text-xs leading-5 font-semibold rounded-full 
                                                <?php echo $signal['type'] === 'buy' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800'; ?>">
                                                <?php echo $signal['type']; ?>
                                            </span>
                                        </td>
                                        <td class="px-6 py-4 text-sm text-gray-900">$<?php echo number_format($signal['price'], 8); ?></td>
                                        <td class="px-6 py-4 text-sm text-gray-500">
                                            <?php echo date('Y-m-d H:i', strtotime($signal['created_at'])); ?>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>

                <!-- توکن‌های برتر -->
                <div class="bg-white rounded-lg shadow">
                    <div class="p-6">
                        <h2 class="text-xl font-bold text-gray-800 mb-4">توکن‌های برتر (حجم معاملات)</h2>
                        <div class="overflow-x-auto">
                            <table class="min-w-full">
                                <thead>
                                    <tr class="text-right bg-gray-50">
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">توکن</th>
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">قیمت</th>
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">حجم 24h</th>
                                        <th class="px-6 py-3 text-sm font-semibold text-gray-600">تغییر 24h</th>
                                    </tr>
                                </thead>
                                <tbody class="divide-y divide-gray-200">
                                    <?php foreach ($top_volume as $token): ?>
                                    <tr>
                                        <td class="px-6 py-4 text-sm font-medium text-gray-900">
                                            <?php echo htmlspecialchars($token['symbol']); ?>
                                        </td>
                                        <td class="px-6 py-4 text-sm text-gray-900">
                                            $<?php echo number_format($token['price'], 8); ?>
                                        </td>
                                        <td class="px-6 py-4 text-sm text-gray-900">
                                            $<?php echo number_format($token['volume_24h']); ?>
                                        </td>
                                        <td class="px-6 py-4 text-sm">
                                            <span class="<?php echo $token['price_change_24h'] >= 0 ? 'text-green-600' : 'text-red-600'; ?>">
                                                <?php echo number_format($token['price_change_24h'], 2); ?>%
                                            </span>
                                        </td>
                                    </tr>
                                    <?php endforeach; ?>
                                </tbody>
                            </table>
                        </div>
                    </div>
                </div>
            </div>

            <!-- گزارش روزانه -->
            <?php if ($latest_report): 
                $report_data = json_decode($latest_report['market_stats'], true);
            ?>
            <div class="bg-white rounded-lg shadow p-6 mb-6">
                <h2 class="text-xl font-bold text-gray-800 mb-4">گزارش روزانه</h2>
                <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                    <div>
                        <h3 class="text-lg font-semibold text-gray-700 mb-2">آمار بازار</h3>
                        <ul class="space-y-2">
                            <li class="text-sm text-gray-600">
                                حجم معاملات: $<?php echo number_format($report_data['total_volume']); ?>
                            </li>
                            <li class="text-sm text-gray-600">
                                تغییر حجم: <?php echo number_format($report_data['volume_change'], 2); ?>%
                            </li>
                            <li class="text-sm text-gray-600">
                                توکن‌های فعال: <?php echo number_format($report_data['active_tokens']); ?>
                            </li>
                        </ul>
                    </div>
                    <!-- بخش‌های دیگر گزارش -->
                </div>
            </div>
            <?php endif; ?>
        </main>
    </div>

    <!-- فوتر -->
    <footer class="bg-white shadow">
        <div class="max-w-7xl mx-auto py-4 px-4 sm:px-6 lg:px-8">
            <p class="text-center text-gray-500 text-sm">
                سیستم مانیتورینگ توکن‌های سولانا © <?php echo date('Y'); ?>
            </p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/chart.js@3.7.0/dist/chart.min.js"></script>
    <script>
        // اضافه کردن نمودارها و عملکردهای اضافی در صورت نیاز
    </script>
</body>
</html>