<?php
namespace TokenMonitor;

class TelegramBot {
    private $token;
    private $api_url = 'https://api.telegram.org/bot';
    private $default_chat_id;
    private $bot_username;
    private $db;

    public function __construct($config, $db) {
        $this->token = $config['telegram']['bot_token'];
        $this->default_chat_id = $config['telegram']['chat_id'];
        $this->bot_username = $config['telegram']['bot_username'];
        $this->api_url .= $this->token;
        $this->db = $db;
    }

    // ارسال پیام به تلگرام
    public function sendMessage($message, $chat_id = null, $parse_mode = 'HTML', $disable_web_page_preview = true) {
        if (empty($chat_id)) {
            $chat_id = $this->default_chat_id;
        }

        $data = [
            'chat_id' => $chat_id,
            'text' => $message,
            'parse_mode' => $parse_mode,
            'disable_web_page_preview' => $disable_web_page_preview
        ];

        return $this->makeRequest('sendMessage', $data);
    }

    // ارسال سیگنال خرید
    public function sendBuySignal($token_data, $analysis) {
        $message = $this->formatBuySignal($token_data, $analysis);
        
        // ذخیره سیگنال در دیتابیس
        $this->saveBuySignal($token_data, $analysis);
        
        return $this->sendMessage($message);
    }

    // فرمت‌بندی پیام سیگنال خرید
    private function formatBuySignal($token_data, $analysis) {
        $message = "🚀 <b>سیگنال خرید جدید</b>\n\n";
        $message .= "🔹 <b>نام توکن:</b> {$token_data['name']}\n";
        $message .= "🔹 <b>نماد:</b> {$token_data['symbol']}\n";
        $message .= "🔹 <b>شبکه:</b> Solana\n\n";

        // اطلاعات قیمت
        $message .= "💰 <b>قیمت فعلی:</b> $" . number_format($token_data['price'], 8) . "\n";
        $message .= "💎 <b>مارکت کپ:</b> $" . number_format($token_data['market_cap']) . "\n";
        $message .= "💧 <b>لیکوئیدیتی:</b> $" . number_format($token_data['liquidity']) . "\n";
        $message .= "📊 <b>حجم 24h:</b> $" . number_format($token_data['volume_24h']) . "\n\n";

        // تحلیل تکنیکال
        $message .= "📈 <b>تحلیل تکنیکال:</b>\n";
        $message .= "• روند روزانه: " . $this->getTrendEmoji($analysis['trend']['daily']) . " " . $analysis['trend']['daily'] . "\n";
        $message .= "• RSI: " . $analysis['technical']['rsi'] . "\n";
        if (isset($analysis['technical']['macd'])) {
            $message .= "• MACD: " . $this->getMACDStatus($analysis['technical']['macd']) . "\n";
        }

        // امتیازها
        $message .= "\n⭐️ <b>امتیازها:</b>\n";
        $message .= "• تکنیکال: " . $this->getScoreStars($analysis['scores']['technical']) . "\n";
        $message .= "• فاندامنتال: " . $this->getScoreStars($analysis['scores']['fundamental']) . "\n";
        $message .= "• اجتماعی: " . $this->getScoreStars($analysis['scores']['social']) . "\n";
        $message .= "• کلی: " . $this->getScoreStars($analysis['scores']['total']) . "\n\n";

        // ریسک‌ها
        if (!empty($analysis['risks'])) {
            $message .= "⚠️ <b>هشدارهای ریسک:</b>\n";
            foreach ($analysis['risks'] as $risk) {
                $message .= "• " . $risk['description'] . "\n";
            }
            $message .= "\n";
        }

        // لینک‌ها
        $message .= "🔍 <b>لینک‌ها:</b>\n";
        $message .= "• <a href='https://dexscreener.com/solana/{$token_data['token_address']}'>DexScreener</a>\n";
        if (!empty($token_data['website'])) {
            $message .= "• <a href='{$token_data['website']}'>Website</a>\n";
        }
        
        // شبکه‌های اجتماعی
        if (!empty($token_data['social'])) {
            $message .= "\n🌐 <b>شبکه‌های اجتماعی:</b>\n";
            if (!empty($token_data['social']['twitter'])) {
                $message .= "• <a href='https://twitter.com/{$token_data['social']['twitter']}'>Twitter</a>\n";
            }
            if (!empty($token_data['social']['telegram'])) {
                $message .= "• <a href='https://t.me/{$token_data['social']['telegram']}'>Telegram</a>\n";
            }
            if (!empty($token_data['social']['discord'])) {
                $message .= "• <a href='{$token_data['social']['discord']}'>Discord</a>\n";
            }
        }

        $message .= "\n⚡️ <b>راهنمای خرید:</b>\n";
        $message .= "1. حتماً از ادرس توکن بالا استفاده کنید\n";
        $message .= "2. قبل از خرید تحقیقات خود را انجام دهید\n";
        $message .= "3. از سرمایه‌ای که توان از دست دادن آن را دارید استفاده کنید\n";
        
        return $message;
    }

    // ارسال هشدار ریسک
    public function sendRiskAlert($token_data, $risks) {
        $message = $this->formatRiskAlert($token_data, $risks);
        return $this->sendMessage($message);
    }

    // فرمت‌بندی هشدار ریسک
    private function formatRiskAlert($token_data, $risks) {
        $message = "⚠️ <b>هشدار ریسک</b>\n\n";
        $message .= "🔹 <b>توکن:</b> {$token_data['name']} ({$token_data['symbol']})\n\n";

        $message .= "🚫 <b>موارد ریسک شناسایی شده:</b>\n";
        foreach ($risks as $risk) {
            $severity_emoji = $this->getRiskSeverityEmoji($risk['severity']);
            $message .= "{$severity_emoji} {$risk['description']}\n";
        }

        $message .= "\n⚡️ <b>توصیه‌ها:</b>\n";
        $message .= "• قبل از هرگونه معامله، تحقیقات کامل انجام دهید\n";
        $message .= "• حجم معاملات خود را کاهش دهید\n";
        $message .= "• از استراتژی‌های مدیریت ریسک استفاده کنید\n";

        return $message;
    }

    // ارسال تحلیل روزانه
    public function sendDailyAnalysis($tokens_data) {
        $message = $this->formatDailyAnalysis($tokens_data);
        return $this->sendMessage($message);
    }

    // فرمت‌بندی تحلیل روزانه
    private function formatDailyAnalysis($tokens_data) {
        $message = "📊 <b>تحلیل روزانه بازار</b>\n\n";

        // آمار کلی
        $total_volume = array_sum(array_column($tokens_data, 'volume_24h'));
        $avg_change = array_sum(array_column($tokens_data, 'price_change_24h')) / count($tokens_data);
        
        $message .= "📈 <b>آمار کلی:</b>\n";
        $message .= "• حجم کل معاملات: $" . number_format($total_volume) . "\n";
        $message .= "• میانگین تغییر قیمت: " . number_format($avg_change, 2) . "%\n\n";

        // توکن‌های برتر
        $message .= "🏆 <b>بهترین عملکردها (24h):</b>\n";
        $top_performers = array_slice($tokens_data, 0, 5);
        foreach ($top_performers as $token) {
            $message .= "• {$token['symbol']}: " . number_format($token['price_change_24h'], 2) . "%\n";
        }

        return $message;
    }

    // ذخیره سیگنال در دیتابیس
    private function saveBuySignal($token_data, $analysis) {
        try {
            $stmt = $this->db->prepare("
                INSERT INTO signals 
                (token_address, type, price, analysis_data, created_at)
                VALUES (?, 'buy', ?, ?, NOW())
            ");
            
            $stmt->execute([
                $token_data['token_address'],
                $token_data['price'],
                json_encode($analysis)
            ]);
            
            return true;
        } catch (\PDOException $e) {
            error_log("Error saving buy signal: " . $e->getMessage());
            return false;
        }
    }

    // تبدیل روند به ایموجی
    private function getTrendEmoji($trend) {
        switch (strtolower($trend)) {
            case 'uptrend':
                return '📈';
            case 'downtrend':
                return '📉';
            default:
                return '➡️';
        }
    }

    // تبدیل امتیاز به ستاره
    private function getScoreStars($score) {
        $stars = round($score / 20); // تبدیل امتیاز 0-100 به 0-5 ستاره
        return str_repeat('⭐', $stars) . str_repeat('☆', 5 - $stars);
    }

    // تبدیل شدت ریسک به ایموجی
    private function getRiskSeverityEmoji($severity) {
        switch (strtolower($severity)) {
            case 'critical':
                return '🔴';
            case 'high':
                return '🟠';
            case 'medium':
                return '🟡';
            default:
                return '🟢';
        }
    }

    // وضعیت MACD
    private function getMACDStatus($macd) {
        if ($macd['histogram'] > 0) {
            return "مثبت (" . number_format($macd['histogram'], 4) . ")";
        } else {
            return "منفی (" . number_format($macd['histogram'], 4) . ")";
        }
    }

    // ارسال درخواست به API تلگرام
    private function makeRequest($method, $data) {
        $url = "{$this->api_url}/{$method}";
        
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => $url,
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_POST => true,
            CURLOPT_POSTFIELDS => http_build_query($data),
            CURLOPT_TIMEOUT => 10
        ]);

        $response = curl_exec($curl);
        $error = curl_error($curl);
        curl_close($curl);

        if ($error) {
            error_log("Telegram API Error: " . $error);
            return false;
        }

        $result = json_decode($response, true);
        if (!$result['ok']) {
            error_log("Telegram API Error: " . $response);
            return false;
        }

        return $result['result'];
    }
}