<?php
namespace TokenMonitor;

class SocialAnalyzer {
    private $db;
    private $config;

    public function __construct($db, $config) {
        $this->db = $db;
        $this->config = $config;
    }

    // تحلیل جامع اطلاعات اجتماعی
    public function analyzeSocialPresence($token_data) {
        $social_data = [
            'twitter' => $this->analyzeTwitter($token_data['social']['twitter']),
            'telegram' => $this->analyzeTelegram($token_data['social']['telegram']),
            'discord' => $this->analyzeDiscord($token_data['social']['discord']),
            'website' => $this->analyzeWebsite($token_data['website']),
            'team' => $this->analyzeTeam($token_data['team']),
            'community' => $this->analyzeCommunityEngagement($token_data)
        ];

        return [
            'data' => $social_data,
            'score' => $this->calculateSocialScore($social_data),
            'recommendations' => $this->generateSocialRecommendations($social_data)
        ];
    }

    // تحلیل توییتر
    private function analyzeTwitter($twitter_data) {
        if (empty($twitter_data['username'])) {
            return [
                'exists' => false,
                'score' => 0,
                'issues' => ['No Twitter presence']
            ];
        }

        $metrics = $this->getTwitterMetrics($twitter_data['username']);
        
        $analysis = [
            'exists' => true,
            'account_age' => $metrics['account_age'],
            'followers' => $metrics['followers'],
            'engagement_rate' => $metrics['engagement_rate'],
            'post_frequency' => $metrics['post_frequency'],
            'sentiment' => $this->analyzeTweetSentiment($metrics['recent_tweets']),
            'suspicious_patterns' => $this->detectSuspiciousPatterns($metrics)
        ];

        // محاسبه امتیاز توییتر
        $score = 0;
        
        // امتیاز سن اکانت
        if ($analysis['account_age'] > 180) { // بیشتر از 6 ماه
            $score += 20;
        } elseif ($analysis['account_age'] > 90) { // بیشتر از 3 ماه
            $score += 10;
        }

        // امتیاز تعداد فالوور
        if ($analysis['followers'] > 10000) {
            $score += 25;
        } elseif ($analysis['followers'] > 5000) {
            $score += 15;
        } elseif ($analysis['followers'] > 1000) {
            $score += 10;
        }

        // امتیاز نرخ تعامل
        if ($analysis['engagement_rate'] > 5) {
            $score += 25;
        } elseif ($analysis['engagement_rate'] > 2) {
            $score += 15;
        }

        // امتیاز فرکانس پست
        if ($analysis['post_frequency'] > 3) { // بیش از 3 پست در روز
            $score += 15;
        } elseif ($analysis['post_frequency'] > 1) {
            $score += 10;
        }

        $analysis['score'] = $score;
        
        return $analysis;
    }

    // تحلیل تلگرام
    private function analyzeTelegram($telegram_data) {
        if (empty($telegram_data['group_url'])) {
            return [
                'exists' => false,
                'score' => 0,
                'issues' => ['No Telegram presence']
            ];
        }

        $metrics = $this->getTelegramMetrics($telegram_data['group_url']);
        
        $analysis = [
            'exists' => true,
            'members' => $metrics['members'],
            'active_members' => $metrics['active_members'],
            'messages_per_day' => $metrics['messages_per_day'],
            'admin_activity' => $metrics['admin_activity'],
            'group_age' => $metrics['group_age'],
            'suspicious_activity' => $this->detectTelegramSpam($metrics)
        ];

        // محاسبه امتیاز تلگرام
        $score = 0;
        
        // امتیاز تعداد اعضا
        if ($analysis['members'] > 5000) {
            $score += 25;
        } elseif ($analysis['members'] > 1000) {
            $score += 15;
        }

        // امتیاز اعضای فعال
        $active_ratio = $analysis['active_members'] / $analysis['members'] * 100;
        if ($active_ratio > 20) {
            $score += 25;
        } elseif ($active_ratio > 10) {
            $score += 15;
        }

        // امتیاز فعالیت ادمین
        if ($analysis['admin_activity'] > 20) { // بیش از 20 پیام در روز
            $score += 15;
        } elseif ($analysis['admin_activity'] > 10) {
            $score += 10;
        }

        $analysis['score'] = $score;
        
        return $analysis;
    }

    // تحلیل دیسکورد
    private function analyzeDiscord($discord_data) {
        if (empty($discord_data['server_id'])) {
            return [
                'exists' => false,
                'score' => 0,
                'issues' => ['No Discord presence']
            ];
        }

        $metrics = $this->getDiscordMetrics($discord_data['server_id']);
        
        $analysis = [
            'exists' => true,
            'members' => $metrics['members'],
            'online_members' => $metrics['online_members'],
            'channels' => $metrics['channels'],
            'activity_level' => $metrics['activity_level'],
            'server_age' => $metrics['server_age'],
            'verification_level' => $metrics['verification_level']
        ];

        // محاسبه امتیاز دیسکورد
        $score = 0;
        
        // امتیاز تعداد اعضا
        if ($analysis['members'] > 3000) {
            $score += 20;
        } elseif ($analysis['members'] > 1000) {
            $score += 15;
        }

        // امتیاز اعضای آنلاین
        $online_ratio = $analysis['online_members'] / $analysis['members'] * 100;
        if ($online_ratio > 15) {
            $score += 20;
        } elseif ($online_ratio > 8) {
            $score += 15;
        }

        // امتیاز سطح فعالیت
        if ($analysis['activity_level'] === 'high') {
            $score += 20;
        } elseif ($analysis['activity_level'] === 'medium') {
            $score += 15;
        }

        $analysis['score'] = $score;
        
        return $analysis;
    }

    // تحلیل وبسایت
    private function analyzeWebsite($website_url) {
        if (empty($website_url)) {
            return [
                'exists' => false,
                'score' => 0,
                'issues' => ['No website']
            ];
        }

        $metrics = $this->getWebsiteMetrics($website_url);
        
        $analysis = [
            'exists' => true,
            'ssl_valid' => $metrics['ssl_valid'],
            'load_time' => $metrics['load_time'],
            'domain_age' => $metrics['domain_age'],
            'content_quality' => $metrics['content_quality'],
            'whitepaper_exists' => $metrics['whitepaper_exists'],
            'mobile_friendly' => $metrics['mobile_friendly'],
            'suspicious_elements' => $this->detectSuspiciousWebsiteElements($metrics)
        ];

        // محاسبه امتیاز وبسایت
        $score = 0;
        
        // امتیاز SSL
        if ($analysis['ssl_valid']) {
            $score += 10;
        }

        // امتیاز سن دامنه
        if ($analysis['domain_age'] > 180) { // بیشتر از 6 ماه
            $score += 15;
        } elseif ($analysis['domain_age'] > 90) {
            $score += 10;
        }

        // امتیاز کیفیت محتوا
        if ($analysis['content_quality'] === 'high') {
            $score += 25;
        } elseif ($analysis['content_quality'] === 'medium') {
            $score += 15;
        }

        // امتیاز وایت‌پیپر
        if ($analysis['whitepaper_exists']) {
            $score += 20;
        }

        $analysis['score'] = $score;
        
        return $analysis;
    }

    // تحلیل تیم
    private function analyzeTeam($team_data) {
        $analysis = [
            'team_size' => count($team_data),
            'verified_members' => 0,
            'linkedin_profiles' => 0,
            'github_activity' => 0,
            'previous_projects' => []
        ];

        foreach ($team_data as $member) {
            if ($member['is_verified']) {
                $analysis['verified_members']++;
            }
            if (!empty($member['linkedin'])) {
                $analysis['linkedin_profiles']++;
            }
            if (!empty($member['github'])) {
                $analysis['github_activity'] += $this->analyzeGithubActivity($member['github']);
            }
            if (!empty($member['previous_projects'])) {
                $analysis['previous_projects'] = array_merge(
                    $analysis['previous_projects'],
                    $this->analyzePreviousProjects($member['previous_projects'])
                );
            }
        }

        // محاسبه امتیاز تیم
        $score = 0;
        
        // امتیاز اندازه تیم
        if ($analysis['team_size'] >= 5) {
            $score += 20;
        } elseif ($analysis['team_size'] >= 3) {
            $score += 15;
        }

        // امتیاز اعضای تایید شده
        $verified_ratio = $analysis['verified_members'] / $analysis['team_size'] * 100;
        if ($verified_ratio > 80) {
            $score += 25;
        } elseif ($verified_ratio > 50) {
            $score += 15;
        }

        // امتیاز پروفایل‌های لینکدین
        $linkedin_ratio = $analysis['linkedin_profiles'] / $analysis['team_size'] * 100;
        if ($linkedin_ratio > 80) {
            $score += 15;
        } elseif ($linkedin_ratio > 50) {
            $score += 10;
        }

        $analysis['score'] = $score;
        
        return $analysis;
    }

    // تحلیل تعامل جامعه
    private function analyzeCommunityEngagement($token_data) {
        $engagement_metrics = $this->getEngagementMetrics($token_data);
        
        $analysis = [
            'community_growth' => $engagement_metrics['growth_rate'],
            'engagement_rate' => $engagement_metrics['engagement_rate'],
            'sentiment_analysis' => $this->analyzeCommunitySentiment($engagement_metrics['messages']),
            'active_hours' => $engagement_metrics['active_hours'],
            'geographical_distribution' => $engagement_metrics['geo_distribution']
        ];

        // محاسبه امتیاز تعامل
        $score = 0;
        
        // امتیاز نرخ رشد
        if ($analysis['community_growth'] > 10) { // رشد بیش از 10% در هفته
            $score += 25;
        } elseif ($analysis['community_growth'] > 5) {
            $score += 15;
        }

        // امتیاز نرخ تعامل
        if ($analysis['engagement_rate'] > 8) {
            $score += 25;
        } elseif ($analysis['engagement_rate'] > 4) {
            $score += 15;
        }

        $analysis['score'] = $score;
        
        return $analysis;
    }

    // محاسبه امتیاز کلی اجتماعی
    private function calculateSocialScore($social_data) {
        $weights = [
            'twitter' => 0.25,
            'telegram' => 0.25,
            'discord' => 0.15,
            'website' => 0.15,
            'team' => 0.20
        ];

        $total_score = 0;
        foreach ($weights as $platform => $weight) {
            if (isset($social_data[$platform]['score'])) {
                $total_score += $social_data[$platform]['score'] * $weight;
            }
        }

        return round($total_score, 2);
    }

    // تولید توصیه‌های اجتماعی
    private function generateSocialRecommendations($social_data) {
        $recommendations = [];

        // توصیه‌ها بر اساس توییتر
        if (isset($social_data['twitter'])) {
            if ($social_data['twitter']['followers'] < 1000) {
                $recommendations[] = 'Increase Twitter following for better visibility';
            }
            if ($social_data['twitter']['engagement_rate'] < 2) {
                $recommendations[] = 'Improve Twitter engagement with community';
            }
        }

        // توصیه‌ها بر اساس تلگرام
        if (isset($social_data['telegram'])) {
            if ($social_data['telegram']['active_members'] / $social_data['telegram']['members'] < 0.1) {
                $recommendations[] = 'Increase Telegram community engagement';
            }
        }

        return $recommendations;
    }

    // تحلیل احساسات توییت‌ها
    private function analyzeTweetSentiment($tweets) {
        // TODO: پیاده‌سازی تحلیل احساسات
        return [
            'positive' => 0,
            'neutral' => 0,
            'negative' => 0
        ];
    }

   // تشخیص الگوهای مشکوک
    private function detectSuspiciousPatterns($metrics) {
        $suspicious_patterns = [];

        // بررسی رشد ناگهانی فالوور
        if (isset($metrics['follower_growth_rate']) && $metrics['follower_growth_rate'] > 50) {
            $suspicious_patterns[] = 'Unusual follower growth rate';
        }

        // بررسی نسبت فالوور به فالووینگ
        if (isset($metrics['followers']) && isset($metrics['following'])) {
            $ratio = $metrics['following'] > 0 ? $metrics['followers'] / $metrics['following'] : 0;
            if ($ratio < 0.1) {
                $suspicious_patterns[] = 'Very low follower to following ratio';
            }
        }

        // بررسی فعالیت مشکوک در لایک‌ها و ری‌توییت‌ها
        if (isset($metrics['engagement_stats'])) {
            $engagement = $metrics['engagement_stats'];
            
            // بررسی نسبت لایک به فالوور
            if (isset($engagement['likes']) && isset($metrics['followers'])) {
                $like_ratio = $metrics['followers'] > 0 ? $engagement['likes'] / $metrics['followers'] : 0;
                if ($like_ratio > 0.8) {
                    $suspicious_patterns[] = 'Unusually high like to follower ratio';
                }
            }

            // بررسی نسبت ریتوییت به لایک
            if (isset($engagement['retweets']) && isset($engagement['likes'])) {
                $rt_ratio = $engagement['likes'] > 0 ? $engagement['retweets'] / $engagement['likes'] : 0;
                if ($rt_ratio > 0.7) {
                    $suspicious_patterns[] = 'Suspicious retweet to like ratio';
                }
            }
        }

        // بررسی الگوی زمانی پست‌ها
        if (isset($metrics['posting_pattern'])) {
            $pattern = $metrics['posting_pattern'];
            
            // پست‌های منظم با فاصله دقیقاً یکسان
            if ($pattern['variance'] < 0.1) {
                $suspicious_patterns[] = 'Too regular posting pattern (possible bot)';
            }

            // پست‌های بیش از حد در ساعات غیرمعمول
            if ($pattern['odd_hours_posts'] > 80) {
                $suspicious_patterns[] = 'High activity during unusual hours';
            }
        }

        // بررسی محتوای تکراری
        if (isset($metrics['content_similarity'])) {
            if ($metrics['content_similarity'] > 0.7) {
                $suspicious_patterns[] = 'High amount of repetitive content';
            }
        }

        // بررسی سن اکانت و میزان فعالیت
        if (isset($metrics['account_age']) && isset($metrics['total_posts'])) {
            $posts_per_day = $metrics['account_age'] > 0 ? 
                            $metrics['total_posts'] / $metrics['account_age'] : 0;
            
            if ($posts_per_day > 50) {
                $suspicious_patterns[] = 'Unusually high posting frequency';
            }
        }

        // بررسی الگوی فالوورها
        if (isset($metrics['follower_analysis'])) {
            $follower_stats = $metrics['follower_analysis'];
            
            // درصد بالای اکانت‌های جدید
            if (isset($follower_stats['new_accounts_ratio']) && 
                $follower_stats['new_accounts_ratio'] > 0.6) {
                $suspicious_patterns[] = 'High percentage of new accounts among followers';
            }

            // درصد بالای اکانت‌های غیرفعال
            if (isset($follower_stats['inactive_ratio']) && 
                $follower_stats['inactive_ratio'] > 0.5) {
                $suspicious_patterns[] = 'High percentage of inactive followers';
            }
        }

        return $suspicious_patterns;
    }
    // دریافت متریک‌های توییتر
    private function getTwitterMetrics($username) {
        $curl = curl_init();
        
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.twitter.com/2/users/by/username/{$username}",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer " . $this->config['twitter_bearer_token']
            ]
        ]);

        $response = curl_exec($curl);
        $data = json_decode($response, true);
        curl_close($curl);

        if (isset($data['data'])) {
            $user_id = $data['data']['id'];
            return $this->getDetailedTwitterMetrics($user_id);
        }

        return [
            'account_age' => 0,
            'followers' => 0,
            'engagement_rate' => 0,
            'post_frequency' => 0,
            'recent_tweets' => []
        ];
    }

    // دریافت متریک‌های دقیق توییتر
    private function getDetailedTwitterMetrics($user_id) {
        $metrics = [
            'account_age' => 0,
            'followers' => 0,
            'engagement_rate' => 0,
            'post_frequency' => 0,
            'recent_tweets' => []
        ];

        // دریافت توییت‌های اخیر
        $curl = curl_init();
        curl_setopt_array($curl, [
            CURLOPT_URL => "https://api.twitter.com/2/users/{$user_id}/tweets?max_results=100",
            CURLOPT_RETURNTRANSFER => true,
            CURLOPT_HTTPHEADER => [
                "Authorization: Bearer " . $this->config['twitter_bearer_token']
            ]
        ]);

        $response = curl_exec($curl);
        $data = json_decode($response, true);
        curl_close($curl);

        if (isset($data['data'])) {
            $metrics['recent_tweets'] = $data['data'];
            
            // محاسبه فرکانس پست
            $tweet_count = count($data['data']);
            $first_tweet_date = strtotime($data['data'][$tweet_count - 1]['created_at']);
            $days_diff = (time() - $first_tweet_date) / 86400;
            $metrics['post_frequency'] = $tweet_count / max(1, $days_diff);
        }

        return $metrics;
    }

    // دریافت متریک‌های تلگرام
    private function getTelegramMetrics($group_url) {
        try {
            // استفاده از Telegram Bot API
            $group_username = $this->extractTelegramUsername($group_url);
            $curl = curl_init();
            
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://api.telegram.org/bot{$this->config['telegram_bot_token']}/getChatMembersCount?chat_id=@{$group_username}",
                CURLOPT_RETURNTRANSFER => true
            ]);

            $response = curl_exec($curl);
            $data = json_decode($response, true);
            curl_close($curl);

            if ($data['ok']) {
                return [
                    'members' => $data['result'],
                    'active_members' => 0, // نیاز به دسترسی بیشتر
                    'messages_per_day' => 0,
                    'admin_activity' => 0,
                    'group_age' => 0
                ];
            }
        } catch (\Exception $e) {
            error_log("Error getting Telegram metrics: " . $e->getMessage());
        }

        return [
            'members' => 0,
            'active_members' => 0,
            'messages_per_day' => 0,
            'admin_activity' => 0,
            'group_age' => 0
        ];
    }

    // دریافت متریک‌های دیسکورد
    private function getDiscordMetrics($server_id) {
        try {
            $curl = curl_init();
            
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://discord.com/api/v9/guilds/{$server_id}",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    "Authorization: Bot " . $this->config['discord_bot_token']
                ]
            ]);

            $response = curl_exec($curl);
            $data = json_decode($response, true);
            curl_close($curl);

            if (isset($data['id'])) {
                return [
                    'members' => $data['approximate_member_count'] ?? 0,
                    'online_members' => $data['approximate_presence_count'] ?? 0,
                    'channels' => count($data['channels'] ?? []),
                    'activity_level' => $this->calculateDiscordActivity($data),
                    'server_age' => $this->calculateServerAge($data['created_at']),
                    'verification_level' => $data['verification_level']
                ];
            }
        } catch (\Exception $e) {
            error_log("Error getting Discord metrics: " . $e->getMessage());
        }

        return [
            'members' => 0,
            'online_members' => 0,
            'channels' => 0,
            'activity_level' => 'low',
            'server_age' => 0,
            'verification_level' => 0
        ];
    }

    // محاسبه سطح فعالیت دیسکورد
    private function calculateDiscordActivity($server_data) {
        // بررسی تعداد پیام‌ها و تعاملات در 24 ساعت گذشته
        $activity_score = 0;
        
        if ($server_data['approximate_presence_count'] / $server_data['approximate_member_count'] > 0.2) {
            $activity_score += 2;
        }
        
        if (count($server_data['channels'] ?? []) > 10) {
            $activity_score += 1;
        }
        
        if ($activity_score >= 3) {
            return 'high';
        } elseif ($activity_score >= 1) {
            return 'medium';
        }
        
        return 'low';
    }

    // تحلیل فعالیت گیت‌هاب
    private function analyzeGithubActivity($github_username) {
        try {
            $curl = curl_init();
            
            curl_setopt_array($curl, [
                CURLOPT_URL => "https://api.github.com/users/{$github_username}/events/public",
                CURLOPT_RETURNTRANSFER => true,
                CURLOPT_HTTPHEADER => [
                    "Authorization: token " . $this->config['github_token'],
                    "User-Agent: Token-Monitor"
                ]
            ]);

            $response = curl_exec($curl);
            $events = json_decode($response, true);
            curl_close($curl);

            if (is_array($events)) {
                $recent_activity = 0;
                $now = time();
                
                foreach ($events as $event) {
                    $event_time = strtotime($event['created_at']);
                    if ($now - $event_time <= 2592000) { // 30 days
                        $recent_activity++;
                    }
                }
                
                return $recent_activity;
            }
        } catch (\Exception $e) {
            error_log("Error analyzing Github activity: " . $e->getMessage());
        }

        return 0;
    }

    // تحلیل پروژه‌های قبلی
    private function analyzePreviousProjects($projects) {
        $analyzed_projects = [];
        
        foreach ($projects as $project) {
            try {
                // دریافت اطلاعات پروژه از CoinGecko یا مشابه
                $project_info = $this->getProjectInfo($project['id']);
                
                $analyzed_projects[] = [
                    'name' => $project['name'],
                    'success_score' => $this->calculateProjectSuccess($project_info),
                    'roi' => $this->calculateProjectROI($project_info),
                    'duration' => $this->calculateProjectDuration($project_info),
                    'community_satisfaction' => $this->analyzeProjectSentiment($project['id'])
                ];
            } catch (\Exception $e) {
                error_log("Error analyzing project {$project['name']}: " . $e->getMessage());
            }
        }
        
        return $analyzed_projects;
    }

    // محاسبه موفقیت پروژه
    private function calculateProjectSuccess($project_info) {
        $success_score = 0;
        
        // بررسی ROI
        if ($project_info['roi'] > 100) {
            $success_score += 30;
        } elseif ($project_info['roi'] > 0) {
            $success_score += 15;
        }
        
        // بررسی طول عمر پروژه
        if ($project_info['duration'] > 365) { // بیش از یک سال
            $success_score += 30;
        } elseif ($project_info['duration'] > 180) {
            $success_score += 15;
        }
        
        // بررسی رضایت جامعه
        if ($project_info['community_satisfaction'] > 80) {
            $success_score += 40;
        } elseif ($project_info['community_satisfaction'] > 60) {
            $success_score += 20;
        }
        
        return $success_score;
    }

    // دریافت متریک‌های تعامل
    private function getEngagementMetrics($token_data) {
        return [
            'growth_rate' => $this->calculateGrowthRate($token_data),
            'engagement_rate' => $this->calculateEngagementRate($token_data),
            'messages' => $this->getCommunityMessages($token_data),
            'active_hours' => $this->getActiveHours($token_data),
            'geo_distribution' => $this->getGeoDistribution($token_data)
        ];
    }

    // استخراج نام کاربری تلگرام
    private function extractTelegramUsername($url) {
        return str_replace(['https://t.me/', '@'], '', $url);
    }
}